<?php
declare(strict_types=1);
require_once __DIR__ . '/../app/bootstrap.php';
require_once __DIR__ . '/../app/evolution.php';
require_once __DIR__ . '/../public/_layout.php';
require_login('admin');

$pdo = db();
$ok = '';
$err = '';

$motoboys = $pdo->query("SELECT id, name FROM users WHERE role='motoboy' AND is_active=1 ORDER BY name")->fetchAll();

function normalize_phone(string $raw): string {
    // Aceita 55 + DDD + número (com ou sem símbolos)
    $d = preg_replace('/\D+/', '', $raw);
    if (str_starts_with($d, '0')) $d = ltrim($d, '0');
    if (!str_starts_with($d, '55')) $d = '55' . $d;
    return $d;
}

function make_track_token(): string {
    return bin2hex(random_bytes(24)); // 48 chars
}

function save_upload(array $file): array {
    $maxMb = (int)env('UPLOAD_MAX_MB', 15);
    if (($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
        return [false, null, 'Falha no upload.'];
    }
    if (($file['size'] ?? 0) > ($maxMb * 1024 * 1024)) {
        return [false, null, "Arquivo excede {$maxMb}MB."];
    }
    $tmp = $file['tmp_name'];
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime = finfo_file($finfo, $tmp);
    finfo_close($finfo);
    $allowed = ['image/jpeg'=>'jpg', 'image/png'=>'png', 'image/webp'=>'webp'];
    if (!isset($allowed[$mime])) {
        return [false, null, 'Envie imagem JPG/PNG/WEBP.'];
    }
    $ext = $allowed[$mime];
    $dir = __DIR__ . '/../storage/uploads';
    if (!is_dir($dir)) mkdir($dir, 0775, true);
    $name = date('Ymd_His') . '_' . bin2hex(random_bytes(6)) . '.' . $ext;
    $path = $dir . '/' . $name;
    if (!move_uploaded_file($tmp, $path)) {
        return [false, null, 'Não foi possível salvar o arquivo.'];
    }
    return [true, 'storage/uploads/'.$name, ''];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    csrf_check();
    try {
        $customer_name = trim((string)($_POST['customer_name'] ?? ''));
        $phone_raw = trim((string)($_POST['customer_phone'] ?? ''));
        $address = trim((string)($_POST['address'] ?? ''));
        $motoboy_id = (int)($_POST['motoboy_id'] ?? 0);

        if ($customer_name === '' || $phone_raw === '' || $address === '') {
            throw new Exception('Preencha nome, telefone e endereço.');
        }

        $phone = normalize_phone($phone_raw);

        // Upsert customer
        $pdo->beginTransaction();
        $st = $pdo->prepare("SELECT id FROM customers WHERE phone=? LIMIT 1");
        $st->execute([$phone]);
        $cust = $st->fetch();

        if ($cust) {
            $cid = (int)$cust['id'];
            $st = $pdo->prepare("UPDATE customers SET name=?, address=?, updated_at=? WHERE id=?");
            $st->execute([$customer_name, $address, now_dt(), $cid]);
        } else {
            $st = $pdo->prepare("INSERT INTO customers (phone, name, address, created_at, updated_at) VALUES (?,?,?,?,?)");
            $st->execute([$phone, $customer_name, $address, now_dt(), now_dt()]);
            $cid = (int)$pdo->lastInsertId();
        }

        $notePath = null;
        $noteOrig = null;
        if (!empty($_FILES['note_image']['name'] ?? '')) {
            [$okUp, $p, $msg] = save_upload($_FILES['note_image']);
            if (!$okUp) throw new Exception($msg);
            $notePath = $p;
            $noteOrig = $_FILES['note_image']['name'] ?? null;
        }

        $token = make_track_token();
        $created_by = (int)($_SESSION['user']['id'] ?? 0);

        $st = $pdo->prepare("INSERT INTO orders
            (customer_id, customer_name, customer_phone, address, note_image_path, note_original_name, track_token, status, created_by, assigned_motoboy_id, created_at)
            VALUES (?,?,?,?,?,?,?,?,?,?,?)");
        $st->execute([$cid, $customer_name, $phone, $address, $notePath, $noteOrig, $token, 'created', $created_by, ($motoboy_id ?: null), now_dt()]);
        $oid = (int)$pdo->lastInsertId();

        $pdo->prepare("INSERT INTO order_events (order_id, event, note, created_at) VALUES (?,?,?,?)")
            ->execute([$oid, 'created', 'Pedido cadastrado', now_dt()]);

        $pdo->commit();

        // Send WhatsApp
        $order = [
            'id'=>$oid,'customer_name'=>$customer_name,'track_token'=>$token
        ];
        $msg = msg_template('created', $order);
        evo_send_text($phone, $msg);

        $ok = "Pedido #{$oid} cadastrado. Link de rastreio: " . env('APP_URL','') . "/rastreio.php?t={$token}";
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        $err = $e->getMessage();
    }
}

page_header('Novo Pedido');
?>
<div class="card">
  <h2>Novo pedido</h2>
  <?php if($ok): ?><div class="notice ok"><?= h($ok) ?></div><?php endif; ?>
  <?php if($err): ?><div class="notice err"><?= h($err) ?></div><?php endif; ?>

  <form method="post" enctype="multipart/form-data">
    <input type="hidden" name="_csrf" value="<?= h(csrf_token()) ?>">
    <div class="row">
      <div>
        <label>Nome do cliente</label>
        <input name="customer_name" required>
      </div>
      <div>
        <label>Telefone (WhatsApp)</label>
        <input name="customer_phone" placeholder="(DDD) 9xxxx-xxxx" required>
      </div>
    </div>

    <label>Endereço completo</label>
    <textarea name="address" required></textarea>

    <div class="row">
      <div>
        <label>Motoboy (opcional)</label>
        <select name="motoboy_id">
          <option value="0">— Não atribuir agora —</option>
          <?php foreach($motoboys as $m): ?>
            <option value="<?= (int)$m['id'] ?>"><?= h($m['name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div>
        <label>Foto da nota (JPG/PNG/WEBP)</label>
        <input type="file" name="note_image" accept="image/*">
      </div>
    </div>

    <div style="margin-top:12px;display:flex;gap:10px;flex-wrap:wrap">
      <button class="btn primary" type="submit">Cadastrar pedido</button>
      <a class="btn" href="/admin/pedidos.php">Voltar</a>
    </div>
  </form>
</div>
<?php page_footer(); ?>
