<?php
declare(strict_types=1);

session_start();

// Compat PHP < 8.0 (cPanel às vezes vem 7.x)
if (!function_exists('str_starts_with')) {
    function str_starts_with(string $haystack, string $needle): bool {
        return $needle === '' || strpos($haystack, $needle) === 0;
    }
}

date_default_timezone_set('America/Sao_Paulo'); // Horário de Brasília

function env_load(string $path): array {
    $vars = [];
    if (!file_exists($path)) return $vars;
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || str_starts_with($line, '#')) continue;
        $pos = strpos($line, '=');
        if ($pos === false) continue;
        $key = trim(substr($line, 0, $pos));
        $val = trim(substr($line, $pos + 1));
        $val = trim($val, "\"'");
        $vars[$key] = $val;
    }
    return $vars;
}

$GLOBALS['ENV'] = array_merge($_ENV, env_load(__DIR__ . '/../.env'));

function env(string $key, $default=null) {
    return $GLOBALS['ENV'][$key] ?? $default;
}

function app_key(): string {
    $k = (string)env('APP_KEY', '');
    if (strlen($k) < 16) {
        // chave mínima para HMAC (em dev)
        $k = 'dev_change_me_please_123456';
    }
    return $k;
}

function db(): PDO {
    static $pdo = null;
    if ($pdo) return $pdo;

    $host = env('DB_HOST','localhost');
    $name = env('DB_NAME','farmacia_delivery');
    $user = env('DB_USER','root');
    $pass = env('DB_PASS','');
    $dsn = "mysql:host={$host};dbname={$name};charset=utf8mb4";
    $pdo = new PDO($dsn, $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
    return $pdo;
}

function redirect(string $to): void {
    header('Location: ' . $to);
    exit;
}

function csrf_token(): string {
    if (empty($_SESSION['_csrf'])) {
        $_SESSION['_csrf'] = bin2hex(random_bytes(16));
    }
    return $_SESSION['_csrf'];
}

function csrf_check(): void {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $t = $_POST['_csrf'] ?? '';
        if (!$t || !hash_equals($_SESSION['_csrf'] ?? '', $t)) {
            http_response_code(403);
            exit('CSRF inválido.');
        }
    }
}

function require_login(string $role): void {
    $u = $_SESSION['user'] ?? null;
    if (!$u || ($u['role'] ?? '') !== $role) {
        redirect('/login.php');
    }
}

function require_moto(): void {
    $u = $_SESSION['user'] ?? null;
    if (!$u || ($u['role'] ?? '') !== 'motoboy') {
        redirect('/motoboy/login.php');
    }
}

function h(string $s): string {
    return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

function json_out($data, int $code=200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

function now_dt(): string {
    return (new DateTime('now', new DateTimeZone('America/Sao_Paulo')))->format('Y-m-d H:i:s');
}

function status_label(string $s): string {
    switch ($s) {
        case 'created':   return 'Pedido cadastrado';
        case 'started':   return 'Entrega iniciada';
        case 'arrived':   return 'Motoboy em frente à residência';
        case 'delivered': return 'Entrega finalizada';
        default:          return $s;
    }
}
