<?php
declare(strict_types=1);
require_once __DIR__ . '/app/bootstrap.php';

$token = trim((string)($_GET['t'] ?? ''));
if ($token === '' || strlen($token) < 20) { http_response_code(400); exit('Token inválido.'); }

$pdo = db();
$st = $pdo->prepare("SELECT id, start_lat, start_lng, dest_lat, dest_lng FROM orders WHERE track_token=? LIMIT 1");
$st->execute([$token]);
$o = $st->fetch();
if (!$o) { http_response_code(404); exit('Pedido não encontrado.'); }

$orderId = (int)$o['id'];
?>
<!doctype html>
<html lang="pt-br">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Mapa</title>
  <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" crossorigin=""/>
  <style>
    html,body{height:100%;margin:0}
    #map{height:100%;width:100%}
  </style>
</head>
<body>
<div id="map"></div>

<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js" crossorigin=""></script>
<script>
const orderId = <?= (int)$orderId ?>;

const map = L.map('map');
L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {maxZoom: 19}).addTo(map);

const mkIcon = (color) => new L.Icon({
  iconUrl: `https://raw.githubusercontent.com/pointhi/leaflet-color-markers/master/img/marker-icon-2x-${color}.png`,
  shadowUrl: 'https://unpkg.com/leaflet@1.9.4/dist/images/marker-shadow.png',
  iconSize: [25, 41], iconAnchor: [12, 41], popupAnchor: [1, -34], shadowSize: [41, 41]
});

const greenIcon = mkIcon('green');
const redIcon   = mkIcon('red');
const blueIcon  = mkIcon('blue');

let startMarker=null, motoMarker=null, destMarker=null;
let routeLine = L.polyline([], {weight:4}).addTo(map);

function fitAll(){
  const layers = [];
  if(startMarker) layers.push(startMarker);
  if(motoMarker) layers.push(motoMarker);
  if(destMarker) layers.push(destMarker);
  layers.push(routeLine);
  const g = L.featureGroup(layers);
  if(g.getLayers().length) map.fitBounds(g.getBounds().pad(0.25));
}

async function fetchGps(){
  const r = await fetch('/api/gps_latest.php?order_id=' + orderId + '&limit=300');
  const j = await r.json();
  if(!j.ok) throw new Error(j.error || 'Falha GPS');
  return j;
}

async function refresh(){
  try{
    const gps = await fetchGps();

    const startLat = parseFloat(gps.order.start_lat || '');
    const startLng = parseFloat(gps.order.start_lng || '');
    const destLat  = parseFloat(gps.order.dest_lat || '');
    const destLng  = parseFloat(gps.order.dest_lng || '');

    if(!isNaN(startLat) && !isNaN(startLng)){
      if(!startMarker) startMarker = L.marker([startLat,startLng], {icon: greenIcon}).addTo(map);
      else startMarker.setLatLng([startLat,startLng]);
    }
    if(!isNaN(destLat) && !isNaN(destLng)){
      if(!destMarker) destMarker = L.marker([destLat,destLng], {icon: blueIcon}).addTo(map);
      else destMarker.setLatLng([destLat,destLng]);
    }

    const last = gps.last;
    if(last){
      const lat = parseFloat(last.lat), lng = parseFloat(last.lng);
      if(!motoMarker) motoMarker = L.marker([lat,lng], {icon: redIcon}).addTo(map);
      else motoMarker.setLatLng([lat,lng]);
    }

    const pts = gps.points || [];
    const latlngs = pts.map(p => [parseFloat(p.lat), parseFloat(p.lng)]).filter(a => !isNaN(a[0]) && !isNaN(a[1]));
    routeLine.setLatLngs(latlngs);

    if(!window.__fitted){
      fitAll();
      if(!map.getCenter().lat) map.setView([-10.6, -37.3], 13);
      window.__fitted = true;
    }
  }catch(e){ console.warn(e); }
}
refresh();
setInterval(refresh, 5000);
</script>
</body>
</html>
